import React from 'react';
import PropTypes from 'prop-types';
import { Router4Compat as Router } from 'react-router-4-compat';
import isString from 'lodash/isString';
import get from 'lodash/get';
import config from './utils/config';
import { isAuthed, hasDomain, histories } from './utils/auth';
import { processError } from './utils/error';
import App from './routes/app';
import { getMenus, getModuleInfo, getModuleLayout } from './data/modules';
import Monk from './routes/main/monk';

const registerModel = (app, model) => {
// eslint-disable-next-line no-underscore-dangle
  if (!(app._models.filter(m => m.namespace === model.namespace).length === 1)) {
    app.model(model);
  }
};

const maybeLogin = async (replace) => {
  if (!(await isAuthed())) {
    return replace('/login');
  }
};

const maybeSwitch = async (replace) => {
  if (!(await hasDomain())) {
    return replace('/domain');
  }
};

const authenticated = async (replace) => {
  await maybeLogin(replace);
  await maybeSwitch(replace);
};

const combinePath = (base, path) => {
  if (!base) {
    return path;
  }
  if (base[base.length - 1] === '/') {
    return `${base}${path}`;
  } else {
    return `${base}/${path}`;
  }
};

const createRoute = async (app, group, basePath) => {
  const { name, showName, modules, children } = group;
  const theFullPath = combinePath(basePath, name);
  // noinspection JSUnusedLocalSymbols
  return {
    path: name,
    fullPath: theFullPath,
    name: showName,
    component: Monk,
    getChildRoutes(nextState, cb) {
      createRoutes(app, modules, children, theFullPath)
        .then((result) => {
          cb(null, result);
        })
        .catch((err) => {
          cb(err, null);
        });
    },
  };
};


const createRoutes = async (app, modules, groups, basePath) => {
  const routes = [];
  if (modules) {
    for (const module of modules) {
      let info;
      let layout;
      if (isString(module)) {
        info = await getModuleInfo(module);
        layout = await getModuleLayout(module);
      } else {
        info = module;
        layout = module.layout;
      }
      const { name, showName, icon, description } = info;
      const route = {
        path: name,
        fullPath: combinePath(basePath, name),
        name: showName,
      };
      if (layout.route) {
        const modelBundle = await import(`./models/main/modules/${layout.route}`);
        registerModel(app, modelBundle.default);
        let routeBundle = await import(`./routes/main/modules/${layout.route}`);
        routeBundle = routeBundle.default || routeBundle;
        route.component = routeBundle;
        if (routeBundle.route) {
          for (const key in routeBundle.route) {
            // noinspection JSUnfilteredForInLoop
            if ({}.hasOwnProperty.call(routeBundle.route, key)) {
              // noinspection JSUnfilteredForInLoop
              route[key] = routeBundle.route[key];
            }
          }
        }
      } else {
        route.component = Monk;
      }
      if (route.onEnter) {
        const onEnter = route.onEnter;
        route.onEnter = (nextState, replace, cb) => {
          if (get(nextState, 'location.pathname') === route.fullPath) {
            histories.pushHistory('module', {
              name,
              showName,
              icon,
              description,
              path: route.fullPath,
            }).then(() => new Promise((resolve, reject) => {
              onEnter(nextState, replace, (err, res) => {
                if (err) {
                  reject(err);
                } else {
                  resolve(res);
                }
              });
            })).then(() => cb()).catch(err => cb(err));
          } else {
            return onEnter(nextState, replace, cb);
          }
        };
      } else {
        route.onEnter = (nextState, replace, cb) => {
          if (get(nextState, 'location.pathname') === route.fullPath) {
            histories.pushHistory('module', {
              name,
              showName,
              icon,
              description,
              path: route.fullPath,
            }).then(() => cb()).catch(err => cb(err));
          } else {
            cb();
          }
        };
      }
      routes.push(route);
    }
  }
  if (groups) {
    for (const group of groups) {
      routes.push(await createRoute(app, group, basePath));
    }
  }
  return routes;
};

function RouterConfig({ history, app }) {
  const routes = [
    {
      path: '/',
      component: App,
      indexRoute: {
        onEnter: (nextState, replace, cb) => {
          histories.getLatest('module').then((latest) => {
            if (latest && config.fastNavigationPage) {
              replace('/fastNav');
            } else {
              replace('/main');
            }
            cb();
          }).catch((err) => {
            cb(err);
          });
        },
      },
      childRoutes: [
        {
          path: 'login',
          getComponent(ignored, cb) {
            Promise.all([
              import('./models/login'),
              import('./routes/login'),
            ]).then(([model, route]) => {
              registerModel(app, model.default);
              cb(null, route.default);
            });
          },
        },
        {
          path: 'domain',
          onEnter: (ignored, replace, cb) => {
            maybeLogin(replace).then(() => cb()).catch(err => cb(err));
          },
          getComponent(ignored, cb) {
            Promise.all([
              import('./models/domain'),
              import('./routes/domain'),
            ]).then(([model, route]) => {
              registerModel(app, model.default);
              cb(null, route.default);
            });
          },
        },
        {
          path: 'main',
          onEnter: (ignored, replace, cb) => {
            authenticated(replace).then(() => cb()).catch(err => cb(err));
          },
          getComponent(ignored, cb) {
            Promise.all([
              import('./models/main'),
              import('./routes/main'),
            ]).then(([model, route]) => {
              registerModel(app, model.default);
              cb(null, route.default);
            });
          },
          getChildRoutes: (nextState, cb) => {
            getMenus()
              .then((menus) => {
                createRoutes(app, [], menus, '/main')
                  .then((result) => {
                    cb(null, result);
                  })
                  .catch((err) => {
                    cb(err, null);
                  });
              })
              .catch((err) => {
                cb(err, null);
              });
          },
        },
      ],
    },
  ];
  if (config.fastNavigationPage) {
    routes[0].childRoutes.push({
      path: 'fastNav',
      onEnter: (ignored, replace, cb) => {
        authenticated(replace).then(() => cb()).catch(err => cb(err));
      },
      getComponent(ignored, cb) {
        Promise.all([
          import(`./models/main/modules/${config.fastNavigationPage}`),
          import(`./routes/main/modules/${config.fastNavigationPage}`),
        ]).then(([model, route]) => {
          registerModel(app, model.default);
          cb(null, route.default);
        });
      },
    });
  }
  return (
    <Router history={history} routes={routes} onError={processError} />
  );
}

RouterConfig.propTypes = {
  history: PropTypes.object.isRequired,
  app: PropTypes.object.isRequired,
};

export default RouterConfig;
