import isPlainObject from 'lodash/isPlainObject';
import isObjectLike from 'lodash/isObjectLike';
import isArray from 'lodash/isArray';
import isNumber from 'lodash/isNumber';
import isBoolean from 'lodash/isBoolean';
import isNull from 'lodash/isNull';
import isUndefined from 'lodash/isUndefined';
import isNil from 'lodash/isNil';
import map from 'lodash/map';
import mapValues from 'lodash/mapValues';
import curry from 'lodash/curry';
import flow from 'lodash/fp/flow';
import toPairs from 'lodash/fp/toPairs';
import flatMap from 'lodash/fp/flatMap';
import _ from 'lodash/fp/placeholder';
import { Resolver } from 'fastjson_ref_resolver';

export function checkStatus(response) {
  if (response.status >= 200 && response.status < 300) {
    return response;
  }
  return response.json().then((data) => {
    const error = new Error(data.message ? data.message : response.statusText);
    error.data = data;
    throw error;
  });
}

export function normParams(unnormed) {
  if (isPlainObject(unnormed)) {
    return flow(
      toPairs,
      flatMap(([k, v]) => (isArray(v) ? v.map(vv => [k, vv]) : [[k, v]])),
    )(unnormed);
  } else {
    return unnormed;
  }
}

// eslint-disable-next-line max-len
export function parseObject(response, middleware, { num2str = false, bool2str = false, nul2str = false, ud2str = false, nil2str = false } = {}) {
  if (response.status === 204) { // no-content
    return null;
  } else {
    const contentType = response.headers.get('content-type');
    if (contentType) {
      const needMap = num2str || bool2str || nul2str || ud2str || nil2str;
      const mapStr = (value) => {
        if (num2str && isNumber(value)) {
          return value.toString();
        }
        if (bool2str && isBoolean(value)) {
          return value.toString();
        }
        if (nul2str && isNull(value)) {
          return '';
        }
        if (ud2str && isUndefined(value)) {
          return '';
        }
        if (nil2str && isNil(value)) {
          return '';
        }
        return value;
      };
      const mapObj = (obj, mapArrFunc) => {
        if (isArray(obj)) {
          return mapArrFunc(obj, mapObj);
        }
        if (isPlainObject(obj)) {
          return mapValues(obj, (val) => {
            const ret = mapStr(val);
            return mapObj(ret, mapArrFunc);
          });
        }
        return obj;
      };
      const mapArr = (arr, mapObjFunc) => {
        if (isPlainObject(arr)) {
          return mapObjFunc(arr, mapArr);
        }
        if (isArray(arr)) {
          return map(arr, (val) => {
            const ret = mapStr(val);
            return mapArr(ret, mapObjFunc);
          });
        }
        return arr;
      };
      const mapValue = curry(mapObj)(_, mapArr);
      if (contentType.indexOf('json') !== -1) {
        return response.json()
          .then((json) => {
            let out = json;
            if (isObjectLike(out)) {
              out = new Resolver(out).resolve();
            }
            return middleware ? middleware(out) : out;
          })
          .then((data) => {
            return needMap ? mapValue(data) : data;
          });
      } else if (contentType.indexOf('xml') !== -1) {
        return Promise.all([response.text(), import('xml2js')])
          .then(([text, xml2js]) => {
            const { parseString } = xml2js;
            return JSON.parse(parseString(text, {}));
          })
          .then((json) => {
            let out = json;
            if (isObjectLike(out)) {
              out = new Resolver(out).resolve();
            }
            return middleware ? middleware(out) : out;
          })
          .then((data) => {
            return needMap ? mapValue(data) : data;
          });
      } else if (contentType.indexOf('text') !== -1) {
        return response.text();
      } else {
        throw new Error(`Unsupported response content type: ${contentType}`);
      }
    } else {
      throw new Error('No response content type.');
    }
  }
}
