import isString from 'lodash/isString';
import get from 'lodash/get';
import flow from 'lodash/fp/flow';
import sortBy from 'lodash/fp/sortBy';
import reverse from 'lodash/fp/reverse';

/**
 * @callback KeyExtractor
 * @param value
 * @return key
 */

/**
 * 获取用于对比的key
 * @param value
 * @param {string|KeyExtractor} [keyMethod]
 * @return {*}
 */
const getKey = (value, keyMethod) => {
  if (!keyMethod) {
    return value;
  } else if (isString(keyMethod)) {
    return get(value, keyMethod);
  } else {
    return keyMethod(value);
  }
};

/**
 * 在数组中查找某个对象，并返回
 * @param {Array} array
 * @param toFind 查找目标
 * @param {string|KeyExtractor} [key] 用于判断2个数据是否相同
 * @return result 找到的对象
 */
const findBy = (array, toFind, key) => {
  if (array) {
    for (const el of array) {
      if (getKey(el, key) === toFind) {
        return el;
      }
    }
  }
  return null;
};

/**
 * 计算频率数据
 * @param {Array} array 数据
 * @param {string|KeyExtractor} [key] 用于判断2个数据是否相同
 * @param {string} [order=desc] 怎么排序，可选值为asc和desc
 */
export const toHistogram = (array, key, order = 'desc') => {
  if (!array || array.length === 0) {
    return [];
  }
  const res = [];
  let newKey;
  if (!key) {
    newKey = 'data';
  } else if (isString(key)) {
    newKey = `data.${key}`;
  } else {
    newKey = value => key(value.data);
  }
  for (const el of array) {
    const find = findBy(res, getKey(el, key), newKey);
    if (find) {
      ++find.num;
    } else {
      res.push({
        data: el,
        num: 1,
      });
    }
  }
  switch (order) {
    case 'desc':
      return flow(
        sortBy(v => v.num),
        reverse,
      )(res);
    case 'asc':
      return sortBy(v => v.num, res);
    default:
      throw new Error(`unsupported order: ${order}`);
  }
};
