import { routerRedux } from 'dva/router';
import { isString } from 'lodash';
import { logout } from '../../services/main';
import { fetchDomains, switchDomain, currentDomain } from '../../services/domain';
import { getMenus, getModuleInfo } from '../../data/modules';
import { fullPath } from '../../utils/helper';
import { delToken, getDomain, setDomain, delDomain, getUser, delUser } from '../../utils/auth';

const createMenu = async (configure) => {
  const menu = {
    name: configure.name,
    icon: configure.icon,
    text: configure.showName,
  };
  if (!menu.children) {
    menu.children = [];
  }
  if (configure.children) {
    for (const child of configure.children) {
      if (child) {
        menu.children.push(await createMenu(child));
      }
    }
  }
  if (configure.modules) {
    for (const module of configure.modules) {
      let info = module;
      if (isString(module)) {
        info = await getModuleInfo(module);
      }
      if (info) {
        menu.children.push({
          name: info.name,
          icon: info.icon,
          text: info.showName,
        });
      }
    }
  }
  return menu;
};

const makeIcon = (menu) => {
  const children = menu.children;
  if (!children || children.length === 0) {
    if (!menu.icon) {
      menu.icon = 'file'; // eslint-disable-line no-param-reassign
    }
  } else {
    if (!menu.icon) {
      menu.icon = 'folder'; // eslint-disable-line no-param-reassign
    }
    for (const child of children) {
      makeIcon(child);
    }
  }
};

const createMenus = async (menusConfigure) => {
  const menus = [];
  if (menusConfigure) {
    for (const configure of menusConfigure) {
      menus.push(await createMenu(configure));
    }
  }
  for (const menu of menus) {
    makeIcon(menu);
  }
  return menus;
};

export default {

  namespace: 'main',

  state: {
    user: '',
    domain: '',
    domainList: [],
    menus: [],
  },

  subscriptions: {
    setup({ dispatch, history }) {  // eslint-disable-line
    },
  },

  effects: {
    *fetchDomain(ignored, { put, call }) {
      const domain = yield call(getDomain);
      let domainName;
      if (!domain) {
        const { path, name } = yield call(currentDomain);
        yield call(setDomain, name, path);
        domainName = name;
      } else {
        domainName = domain.name;
      }
      yield put({ type: 'switchDomainSuccess', payload: domainName });
    },
    *fetchDomains(ignored, { put, call }) {
      const list = yield call(fetchDomains, '/', true);
      yield put({ type: 'queryDomainsSuccess', payload: list });
    },
    *switchDomain({ payload: domainId }, { put, call }) {
      yield call(switchDomain, domainId);
      const { path, name } = yield call(currentDomain);
      yield call(setDomain, name, path);
      yield put({ type: 'switchDomainSuccess', payload: name });
      yield put(routerRedux.push(fullPath('/main')));
    },
    *fetchUser(ignored, { put, call }) {
      const user = yield call(getUser);
      if (!user) {
        yield put(routerRedux.push(fullPath('/login')));
      }
      yield put({ type: 'queryUserSuccess', payload: user.name });
    },
    *fetchModules(ignored, { put, call }) {
      const configures = yield call(getMenus);
      const menus = yield call(createMenus, configures);
      yield put({ type: 'queryMenusSuccess', payload: menus });
    },
    *logout(ignored, { put, call }) {
      yield call(logout);
      yield call(delToken);
      yield call(delUser);
      yield call(delDomain);
      yield put(routerRedux.push(fullPath('/login')));
    },
  },

  reducers: {
    queryUserSuccess(state, { payload: user }) {
      return {
        ...state,
        user,
      };
    },
    queryDomainsSuccess(state, { payload: list }) {
      return {
        ...state,
        domainList: list,
      };
    },
    switchDomainSuccess(state, { payload: domain }) {
      return {
        ...state,
        domain,
      };
    },
    queryMenusSuccess(state, { payload: menus }) {
      return {
        ...state,
        menus,
      };
    },
  },

};
