import { browserHistory } from 'dva/router';
import resolvePathname from 'resolve-pathname';
import { isString } from 'lodash';
import config from '../utils/config';

const { contextPath } = config;

const makePath = (base, path, withContext = true) => {
  if (path.startsWith('/')) {
    return withContext ? `${contextPath}${path}` : path;
  }
  const basePath = base.endsWith('/') ? base : `${base}/`;
  return resolvePathname(path, basePath);
};

const processPath = (base, path, withContext = true) => {
  if (isString(path)) {
    return makePath(base, path, withContext);
  }
  return {
    ...path,
    pathname: makePath(base, path.pathname, withContext),
  };
};

const getHistoryBase = (history) => {
  if (history.location) {
    return history.location.pathname;
  } else {
    return location.pathname;
  }
};

export const history = browserHistory;
export const location = {};

history.listen((loc) => {
  location.pathname = loc.pathname;
  location.state = loc.state;
  location.search = loc.search;
  location.hash = loc.hash;
});

export const push = (path, state, withContext = true) => {
  return history.push(processPath(getHistoryBase(history), path, withContext), state);
};

export const replace = (path, state, withContext = true) => {
  return history.replace(processPath(getHistoryBase(history), path, withContext), state);
};

export const go = (n) => {
  return history.go(n);
};

export const goBack = () => {
  return history.goBack();
};

export const goForward = () => {
  return history.goForward();
};


const checkThis = (theThis) => {
  if (!theThis || !theThis.props) {
    throw new Error('The this is not a component.');
  }
  if (!theThis.props.router) {
    throw new Error('please use withRouter.');
  }
};

const getThisBase = (theThis) => {
  if (theThis.props.location) {
    return theThis.props.location.pathname;
  } else if (typeof window !== 'undefined') {
    return window.location.pathname; // eslint-disable-line no-undef
  } else {
    throw new Error('can not find base path!');
  }
};

export const thisPush = (theThis, pathOrLoc, withContext = true) => {
  checkThis(theThis);
  const route = processPath(getThisBase(theThis), pathOrLoc, withContext);
  return theThis.props.router.push(route);
};

export const thisReplace = (theThis, pathOrLoc, withContext = true) => {
  checkThis(theThis);
  const route = processPath(getThisBase(theThis), pathOrLoc, withContext);
  return theThis.props.router.replace(route);
};

export const thisGo = (theThis, n) => {
  checkThis(theThis);
  return theThis.props.router.go(n);
};

export const thisGoBack = (theThis) => {
  checkThis(theThis);
  return theThis.props.router.goBack();
};

export const thisGoForward = (theThis) => {
  checkThis(theThis);
  return theThis.props.router.goForward();
};
