import AXO from 'axo';
import bowser from 'bowser';

export const isBrowserSupport = () => {
  return bowser.msie;
};

/**
 * 初始化
 * @return {number} 0: success; 1: blocked by the browser; 2: driver not installed
 */
export const init = () => {
  if (bowser.msie) {
    try {
      const se = new AXO('SafeEngineCOM.SafeEngineCtl');
      if (se) {
        return 0;
      } else {
        return 2;
      }
    } catch (err) {
      return 1;
    }
  } else {
    return 3;
  }
};

/**
 * 获取证书
 * @param password usb-key的密钥
 * @return {{[cert]: string, [deadTime]: number, status: number, [error]: string}}
 */
export const getCert = (password) => {
  if (bowser.msie) {
    const safeEngine = new AXO('SafeEngineCOM.SafeEngineCtl');
    if (!safeEngine) {
      return {
        status: 1,
        input: password,
        error: '未安装USB证书控件。',
      };
    }
    safeEngine.SEH_InitialSession(27, 'com1', password, 100, 2, '', '');
    if (safeEngine.ErrorCode !== 0) {
      return {
        status: safeEngine.ErrorCode,
        input: password,
        error: 'USB-KEY初始化失败，请确认USB-KEY是否插入或密码是否正确。',
      };
    }
    try {
      const cert = safeEngine.SEH_GetSelfCertificate(27, 'com1', '');
      if (safeEngine.ErrorCode !== 0) {
        return {
          status: safeEngine.ErrorCode,
          input: password,
          error: `获取个人证书错误，错误代码为：${safeEngine.ErrorCode}。`,
        };
      }
      const res = {
        status: 0,
        cert,
        input: password,
      };
      const deadTime = safeEngine.SEH_GetCertValidDate(cert);
      res.deadTime = parseInt(deadTime, 10);
      return res;
    } finally {
      safeEngine.SEH_ClearSession();
    }
  } else {
    return {
      status: 1,
      error: '当前浏览器不支持activeX控件，请切换ie或带ie内核的360之类的浏览器。',
    };
  }
};

/**
 * 签名
 * @param password usb-key的密钥
 * @param data 需要签名的数据，从服务端获取
 * @return {{input: string, [cert]: string, [deadTime]: number, [signed]: string, status: number, [error]: string}}
 */
export const sign = (password, data) => {
  if (bowser.msie || bowser.msedge) {
    const safeEngine = new AXO('SafeEngineCOM.SafeEngineCtl');
    if (!safeEngine) {
      return {
        status: 1,
        input: password,
        error: '未安装USB证书控件。',
      };
    }
    safeEngine.SEH_InitialSession(27, 'com1', password, 100, 2, '', '');
    if (safeEngine.ErrorCode !== 0) {
      return {
        status: safeEngine.ErrorCode,
        input: password,
        error: 'USB-KEY初始化失败，请确认USB-KEY是否插入或密码是否正确。',
      };
    }
    try {
      const cert = safeEngine.SEH_GetSelfCertificate(27, 'com1', '');
      if (safeEngine.ErrorCode !== 0) {
        return {
          status: safeEngine.ErrorCode,
          input: password,
          error: `获取个人证书错误，错误代码为：${safeEngine.ErrorCode}。`,
        };
      }
      const res = { cert, status: 0, input: password };
      const deadTime = safeEngine.SEH_GetCertValidDate(cert);
      res.deadTime = parseInt(deadTime, 10);
      const signed = safeEngine.SEH_SignData(data, 3);
      if (safeEngine.ErrorCode !== 0) {
        return {
          status: safeEngine.ErrorCode,
          input: password,
          error: `数字签名失败，错误代码为：${safeEngine.ErrorCode}。`,
        };
      }
      res.signed = signed;
      return res;
    } finally {
      safeEngine.SEH_ClearSession();
    }
  } else {
    return {
      status: 1,
      error: '当前浏览器不支持activeX控件，请切换ie或带ie内核的360之类的浏览器。',
    };
  }
};
