import { routerRedux } from 'dva/router';
import { authorize, login, userInfo } from '../services/login';
import { validate as passValidate } from '../services/login/password';
import { requestCode, validate as ucaValidate } from '../services/login/uca';
import { fullPath, encrypt } from '../utils/helper';
import { setToken, setUser, setDomain, histories } from '../utils/auth';
import { switchDomain, currentDomain } from '../services/domain';
import { errors } from '../utils/error';
import config from '../utils/config';
import { getStore } from '../index';

const successAuthed = async (tokenId) => {
  await setToken(tokenId);
  const uInfo = await userInfo();
  await setUser(uInfo.id, uInfo.name);
  const path = await histories.getLatest('domain');
  if (!path) {
    getStore().dispatch(routerRedux.push(fullPath('/domain')));
  } else {
    await switchDomain(path);
    const domain = await currentDomain();
    if (domain) {
      await setDomain(domain.name, path);
      const latest = await histories.getLatest('module');
      if (latest && config.fastNavigationPage) {
        getStore().dispatch(routerRedux.push(fullPath('/fastNav')));
      } else {
        getStore().dispatch(routerRedux.push(fullPath('/main')));
      }
    } else {
      getStore().dispatch(routerRedux.push(fullPath('/domain')));
    }
  }
};

const processAuthRequirements = (requirements, supports) => {
  let res = [];
  for (const requirement of requirements) {
    const { authTypes } = requirement;
    if (authTypes && authTypes.length > 0) {
      const filteredAuthTypes = authTypes.filter(authType => supports.indexOf(authType) !== -1);
      if (filteredAuthTypes.length === 0) {
        throw errors.unsupportedAuthType(authTypes);
      }
      res = res.map(require => require.filter(item => authTypes.indexOf(item) === -1)).filter(require => require.length > 0);
      res.push(filteredAuthTypes);
    }
  }
  return res;
};

let tkId;

export default {
  namespace: 'login',
  state: {
    status: 'login',
    userName: '',
    ucaCode: '',
    authRequires: [],
  },
  reducers: {
    setStatus(state, { payload }) {
      return {
        ...state,
        status: payload,
      };
    },
    setUserName(state, { payload }) {
      return {
        ...state,
        userName: payload,
      };
    },
    setUCACode(state, { payload }) {
      return {
        ...state,
        ucaCode: payload,
      };
    },
    setAuthRequires(state, { payload }) {
      return {
        ...state,
        authRequires: payload,
      };
    },
  },
  effects: {
    *init(ignored, { put }) {
      yield put({ type: 'setStatus', payload: 'login' });
      yield put({ type: 'setUCACode', payload: '' });
      yield put({ type: 'setAuthRequires', payload: [] });
    },
    *login({ payload: userName }, { call, put }) {
      yield put({ type: 'setUserName', userName });
      const { tokenId, remainedAuthRequirements } = yield call(login, {
        type: 'userName',
        data: userName,
      });
      tkId = tokenId;
      const { requirements } = remainedAuthRequirements;
      const requires = processAuthRequirements(requirements, ['password', 'uca']);
      if (requires.length === 0) {
        yield call(successAuthed, tokenId);
      } else {
        if (requires.some(req => req.indexOf('uca') !== -1)) {
          yield put({ type: 'requestUCACode' });
        }
        yield put({ type: 'setAuthRequires', payload: requires });
      }
      yield put({ type: 'setStatus', payload: 'auth' });
    },
    *requestUCACode(ignored, { call, put }) {
      const { data } = yield call(authorize, yield call(requestCode, encrypt(tkId)));
      yield put({ type: 'setUCACode', payload: data });
    },
    *auth({ payload: { password, uca } }, { call }) {
      let response;
      if (password) {
        response = yield call(authorize, yield call(passValidate, password, encrypt(tkId)));
      } else if (uca) {
        if (uca.status === 0) {
          response = yield call(authorize, yield call(ucaValidate, uca.signed, encrypt(tkId)));
        } else {
          throw errors.generalError(uca.error);
        }
      }
      if (response) {
        const { status, remainedAuthRequirements, data } = response;
        if (status !== 'authed' && status !== 'skipped') {
          throw errors.authFailed(data);
        }
        if (remainedAuthRequirements.requirements.length === 0) {
          yield call(successAuthed, tkId);
        }
      }
    },
/*    *login({ payload }, { call, put }) {
      const loginRequest = {
        type: 'userName',
        data: payload.userName,
        authRequest: yield call(validate, payload.password),
      };
      const result = yield call(login, loginRequest);
      const { tokenId, authResponse, remainedAuthRequirements } = result;
      if (authResponse.status !== 'authed' && authResponse.status !== 'skipped') {
        throw errors.wrongPassword();
      }
      const { requirements } = remainedAuthRequirements;
      if (requirements.length > 0) {
        throw errors.unsupportedAuthType(requirements);
      }
      yield call(setToken, tokenId);
      const uInfo = yield call(userInfo);
      yield call(setUser, uInfo.id, uInfo.name);
      const path = yield call(histories.getLatest, 'domain');
      if (!path) {
        yield put(routerRedux.push(fullPath('/domain')));
      } else {
        yield call(switchDomain, path);
        const domain = yield call(currentDomain);
        if (domain) {
          yield call(setDomain, domain.name, path);
          const latest = yield call(histories.getLatest, 'module');
          if (latest && config.fastNavigationPage) {
            yield put(routerRedux.push(fullPath('/fastNav')));
          } else {
            yield put(routerRedux.push(fullPath('/main')));
          }
        } else {
          yield put(routerRedux.push(fullPath('/domain')));
        }
      }
    },*/
  },
  subscriptions: {},
};
