import React from 'react';
import PropTypes from 'prop-types';
import { Router } from 'dva/router';
import _ from 'lodash';
import config from './utils/config';
import { isAuthed, hasDomain, histories } from './utils/auth';
import { fullPath, makePromise0 } from './utils/helper';
import { processError } from './utils/error';
import App from './routes/app';
import { getMenus, getModuleInfo, getModuleLayout } from './data/modules';
import Monk from './routes/main/monk';

const { contextPath } = config;

const registerModel = (app, model) => {
// eslint-disable-next-line no-underscore-dangle
  if (!(app._models.filter(m => m.namespace === model.namespace).length === 1)) {
    app.model(model);
  }
};

const maybeLogin = async (replace) => {
  if (!(await isAuthed())) {
    return replace(fullPath('/login'));
  }
};

const maybeSwitch = async (replace) => {
  if (!(await hasDomain())) {
    return replace(fullPath('/domain'));
  }
};

const authenticated = async (replace) => {
  await maybeLogin(replace);
  await maybeSwitch(replace);
};

const combinePath = (base, path) => {
  if (!base) {
    return path;
  }
  if (base[base.length - 1] === '/') {
    return `${base}${path}`;
  } else {
    return `${base}/${path}`;
  }
};

const createRoute = async (app, group, basePath) => {
  const { name, showName, modules, children } = group;
  const theFullPath = combinePath(basePath, name);
  // noinspection JSUnusedLocalSymbols
  return {
    path: name,
    fullPath: theFullPath,
    name: showName,
    component: Monk,
    getChildRoutes(nextState, cb) {
      createRoutes(app, modules, children, theFullPath)
        .then((result) => {
          cb(null, result);
        })
        .catch((err) => {
          cb(err, null);
        });
    },
  };
};


const createRoutes = async (app, modules, groups, basePath) => {
  const routes = [];
  if (modules) {
    for (const module of modules) {
      let info;
      let layout;
      if (_.isString(module)) {
        info = await getModuleInfo(module);
        layout = await getModuleLayout(module);
      } else {
        info = module;
        layout = module.layout;
      }
      const { name, showName, icon, description } = info;
      const route = {
        path: name,
        fullPath: combinePath(basePath, name),
        name: showName,
      };
      if (layout.route) {
        const modelBundle = await makePromise0(require('bundle!./models/main/modules/' + layout.route + '/index.js')); // eslint-disable-line import/no-dynamic-require, prefer-template, global-require
        registerModel(app, modelBundle);
        const routeBundle = await makePromise0(require('bundle!./routes/main/modules/' + layout.route + '/index.js')); // eslint-disable-line import/no-dynamic-require, prefer-template, global-require
        route.component = routeBundle;
        if (routeBundle.route) {
          for (const key in routeBundle.route) {
            if ({}.hasOwnProperty.call(routeBundle.route, key)) {
              route[key] = routeBundle.route[key];
            }
          }
        }
      } else {
        route.component = Monk;
      }
      if (route.onEnter) {
        const onEnter = route.onEnter;
        route.onEnter = (nextState, replace, cb) => {
          if (nextState && nextState.location && nextState.location.pathname === route.fullPath) {
            histories.pushHistory('module', {
              name,
              showName,
              icon,
              description,
              path: route.fullPath,
            }).then(() => new Promise((resolve, reject) => {
              onEnter(nextState, replace, (err, res) => {
                if (err) {
                  reject(err);
                } else {
                  resolve(res);
                }
              });
            })).then(() => cb()).catch(err => cb(err));
          } else {
            return onEnter(nextState, replace, cb);
          }
        };
      } else {
        route.onEnter = (nextState, replace, cb) => {
          if (nextState && nextState.location && nextState.location.pathname === route.fullPath) {
            histories.pushHistory('module', {
              name,
              showName,
              icon,
              description,
              path: route.fullPath,
            }).then(() => cb()).catch(err => cb(err));
          } else {
            cb();
          }
        };
      }
      routes.push(route);
    }
  }
  if (groups) {
    for (const group of groups) {
      routes.push(await createRoute(app, group, basePath));
    }
  }
  return routes;
};

function RouterConfig({ history, app }) {
  const routes = [
    {
      path: `${contextPath}/`,
      component: App,
      indexRoute: {
        onEnter: (nextState, replace, cb) => {
          histories.getLatest('module').then((latest) => {
            if (latest && config.fastNavigationPage) {
              replace(`${contextPath}/fastNav`);
            } else {
              replace(`${contextPath}/main`);
            }
            cb();
          }).catch((err) => {
            cb(err);
          });
        },
      },
      childRoutes: [
        {
          path: 'login',
          getComponent(ignored, cb) {
            require.ensure([], (require) => {
              registerModel(app, require('./models/login'));
              cb(null, require('./routes/login'));
            }, 'login');
          },
        },
        {
          path: 'domain',
          onEnter: (ignored, replace, cb) => {
            maybeLogin(replace).then(() => cb()).catch(err => cb(err));
          },
          getComponent(ignored, cb) {
            require.ensure([], (require) => {
              registerModel(app, require('./models/domain'));
              cb(null, require('./routes/domain'));
            }, 'domain');
          },
        },
        {
          path: 'main',
          onEnter: (ignored, replace, cb) => {
            authenticated(replace).then(() => cb()).catch(err => cb(err));
          },
          getComponent(ignored, cb) {
            require.ensure([], (require) => {
              registerModel(app, require('./models/main'));
              cb(null, require('./routes/main'));
            }, 'main');
          },
          getChildRoutes: (nextState, cb) => {
            getMenus()
              .then((menus) => {
                createRoutes(app, [], menus, `${contextPath}/main`)
                  .then((result) => {
                    cb(null, result);
                  })
                  .catch((err) => {
                    cb(err, null);
                  });
              })
              .catch((err) => {
                cb(err, null);
              });
          },
        },
      ],
    },
  ];
  if (config.fastNavigationPage) {
    routes[0].childRoutes.push({
      path: 'fastNav',
      onEnter: (ignored, replace, cb) => {
        authenticated(replace).then(() => cb()).catch(err => cb(err));
      },
      getComponent(ignored, cb) {
        require.ensure([], (require) => {
          registerModel(app, require('./models/main/modules/' + config.fastNavigationPage)); // eslint-disable-line import/no-dynamic-require, prefer-template
          cb(null, require('./routes/main/modules/' + config.fastNavigationPage)); // eslint-disable-line import/no-dynamic-require, prefer-template
        }, 'fastNav');
      },
    });
  }
  return (
    <Router history={history} routes={routes} onError={processError} />
  );
}

RouterConfig.propTypes = {
  history: PropTypes.object,
  app: PropTypes.object,
};

export default RouterConfig;
