import React from 'react';
import { partial } from 'lodash';
import { storiesOf } from '@storybook/react';

import ReactJson from 'react-json-view';
import { DatePicker, Form, Input, Modal, Button } from 'antd';

import dva from 'dva';
import createLoading from 'dva-loading';
import DsTable from '../src/components/table/dstable';
import TableInput from '../src/components/table/input-table';
import UCA from '../src/components/uca';

import { makeCancelable } from '../src/utils/promise';
import { login } from '../src/services/login';
import { validate } from '../src/services/login/password';
import { switchDomain } from '../src/services/domain';
import { setToken } from '../src/utils/auth';

const loginIt = async (userName, password, domainPath) => {
  const result = await login({ type: 'userName', data: userName, authRequest: await validate(password) });
  const { tokenId } = result;
  await setToken(tokenId);
  await switchDomain(domainPath);
};

const wrap = () => Comp => (props) => {
  const app = dva({});
  app.use(createLoading({
    effects: true,
  }));
  const { children, ...rest } = props;
  app.router(() => <Comp app={app} {...rest}>{ children }</Comp>);
  return app.start()();
};

const lazy = action => (Comp) => {
  class Lazy extends React.Component {

    constructor(props, context) {
      super(props, context);
      this.state = {
        loading: true,
      };
    }
    componentDidMount() {
      this.action = makeCancelable(action());
      this.action.run(promise => promise.then(() => {
        this.setState({
          loading: false,
        });
      }));
    }
    componentWillUnmount() {
      this.action.cancel();
    }

    render() {
      if (this.state.loading) {
        return <div>加载中...</div>;
      } else {
        const { children, ...rest } = this.props;
        return (
          <Comp {...rest}>
            { children }
          </Comp>
        );
      }
    }
  }
  return Lazy;
};

storiesOf('a', module)
  .add('1', () => {
    const Comp = lazy(partial(loginIt, 'admin', 'admin', '/'))(wrap()(DsTable));
    const coordinate = {
      containerType: 'module',
      containerName: 'test',
      datasourceName: 'QueryLeftHousesReturnTable',
    };
    return <Comp coordinate={coordinate} />;
  });

const RangePicker = DatePicker.RangePicker;
storiesOf('antd', module)
  .add('RangePicker', () => {
    return <RangePicker />;
  })
  .add('uca', () => {
    return <UCA onChange={evt => console.log(evt)} />;
  });

storiesOf('table-input', module)
  .add('1', () => {
    const Temp = ({ form }) => {
      const props = {
        items: [
          {
            name: 'item0',
            label: 'test0',
          },
          {
            name: 'item1',
            label: 'test1',
          },
        ],
      };
      const handleSubmit = (e) => {
        e.preventDefault();
        form.validateFields((err, values) => {
          if (!err) {
            Modal.info({
              title: '提交内容',
              content: <ReactJson src={values} />,
            });
          }
        });
      };
      return (
        <Form onSubmit={handleSubmit}>
          {
            form.getFieldDecorator('table-input')(
              <TableInput {...props}>
                <Input />
                <Input />
              </TableInput>,
            )
          }
          <Form.Item>
            <Button htmlType="submit">提交</Button>
          </Form.Item>
        </Form>
      );
    };
    const Example = Form.create()(Temp);
    return <Example />;
  });
