import { routerRedux } from 'dva/router';
import { authorize, login, userInfo } from '../services/login';
import { validate as passValidate } from '../services/login/password';
import { requestCode, validate as ucaValidate } from '../services/login/uca';
import { encrypt } from '../utils/helper';
import { setToken, setUser, setDomain, histories } from '../utils/auth';
import { switchDomain, currentDomain } from '../services/domain';
import { errors } from '../utils/error';
import config from '../utils/config';
import { getStore } from '../index';

const successAuthed = async (tokenId, userName, remember) => {
  await histories.pushHistory('userName', null, remember ? userName : '');
  await setToken(tokenId);
  const uInfo = await userInfo();
  await setUser(uInfo.id, uInfo.name);
  const path = await histories.getLatest('domain', uInfo.id);
  if (!path) {
    getStore().dispatch(routerRedux.push('/domain'));
  } else {
    await switchDomain(path);
    const domain = await currentDomain();
    if (domain) {
      await setDomain(domain.name, path);
      const latest = await histories.getLatest('module', uInfo.id);
      if (latest && config.fastNavigationPage) {
        getStore().dispatch(routerRedux.push('/fastNav'));
      } else {
        getStore().dispatch(routerRedux.push('/main'));
      }
    } else {
      getStore().dispatch(routerRedux.push('/domain'));
    }
  }
};

const processAuthRequirements = (requirements, supports) => {
  let res = [];
  for (const requirement of requirements) {
    const { authTypes } = requirement;
    if (authTypes && authTypes.length > 0) {
      const filteredAuthTypes = authTypes.filter(authType => supports.indexOf(authType) !== -1);
      if (filteredAuthTypes.length === 0) {
        throw errors.unsupportedAuthType(authTypes);
      }
      res = res.map(require => require.filter(item => authTypes.indexOf(item) === -1)).filter(require => require.length > 0);
      res.push(filteredAuthTypes);
    }
  }
  return res;
};

let tkId;

export default {
  namespace: 'login',
  state: {
    status: 'login',
    userName: '',
    ucaCode: '',
    authRequires: [],
  },
  reducers: {
    setStatus(state, { payload }) {
      return {
        ...state,
        status: payload,
      };
    },
    setUserName(state, { payload }) {
      return {
        ...state,
        userName: payload,
      };
    },
    setUCACode(state, { payload }) {
      return {
        ...state,
        ucaCode: payload,
      };
    },
    setAuthRequires(state, { payload }) {
      return {
        ...state,
        authRequires: payload,
      };
    },
  },
  effects: {
    *init(ignored, { put, call }) {
      yield put({ type: 'setStatus', payload: 'login' });
      yield put({ type: 'setUCACode', payload: '' });
      yield put({ type: 'setAuthRequires', payload: [] });
      const userName = yield call(histories.getLatest, 'userName', null);
      yield put({ type: 'setUserName', payload: userName || '' });
    },
    *login({ payload: userName }, { call, put }) {
      const { tokenId, remainedAuthRequirements } = yield call(login, {
        type: 'userName',
        data: userName,
      });
      tkId = tokenId;
      const { requirements } = remainedAuthRequirements;
      const requires = processAuthRequirements(requirements, ['password', 'uca']);
      if (requires.length === 0) {
        yield call(successAuthed, tokenId);
      } else {
        if (requires.some(req => req.indexOf('uca') !== -1)) {
          yield put({ type: 'requestUCACode' });
        }
        yield put({ type: 'setAuthRequires', payload: requires });
      }
      yield put({ type: 'setStatus', payload: 'auth' });
    },
    *requestUCACode(ignored, { call, put }) {
      const { data } = yield call(authorize, yield call(requestCode, encrypt(tkId)));
      yield put({ type: 'setUCACode', payload: data });
    },
    *auth({ payload: { userName, password, uca, remember } }, { call }) {
      let response;
      if (password) {
        response = yield call(authorize, yield call(passValidate, password, encrypt(tkId)));
        const { status } = response;
        if (status !== 'authed' && status !== 'skipped') {
          throw errors.wrongPassword();
        }
      }
      if (uca) {
        if (uca.status === 0) {
          response = yield call(authorize, yield call(ucaValidate, uca.signed, encrypt(tkId)));
          const { status } = response;
          if (status !== 'authed' && status !== 'skipped') {
            throw errors.generalError('证书验证失败。');
          }
        } else {
          throw errors.generalError(uca.error);
        }
      }
      if (response) {
        const { status, remainedAuthRequirements, data } = response;
        if (status !== 'authed' && status !== 'skipped') {
          throw errors.authFailed(data);
        }
        if (remainedAuthRequirements.requirements.length === 0) {
          yield call(successAuthed, tkId, userName, remember);
        }
      }
    },
  },
  subscriptions: {},
};
