import createBrowserHistory from 'history/createBrowserHistory';
import isString from 'lodash/isString';
import { makePath } from '../utils/helper';

const processPath = (base, path, withContext = false) => {
  if (isString(path)) {
    return makePath(base, path, withContext);
  }
  return {
    ...path,
    pathname: makePath(base, path.pathname, withContext),
  };
};

const getHistoryBase = (history) => {
  if (history.location) {
    return history.location.pathname;
  } else {
    return location.pathname;
  }
};

let history;
let location;

export const getHistory = (...args) => {
  return history || createHistory(...args);
};
export const getLocation = () => {
  return location || {};
};
export const destroyHistory = () => {
  if (history && history.unlisten) {
    history.unlisten();
    history = null;
  }
};
export const createHistory = (...args) => {
  destroyHistory();
  history = createBrowserHistory(...args);
  // noinspection JSUnresolvedFunction
  history.unlisten = history.listen((loc) => {
    location = { ...loc };
  });
  return history;
};

export const push = (path, state) => {
  return history.push(processPath(getHistoryBase(history), path, false), state);
};

export const replace = (path, state) => {
  return history.replace(processPath(getHistoryBase(history), path, false), state);
};

export const go = (n) => {
  return history.go(n);
};

export const goBack = () => {
  return history.goBack();
};

export const goForward = () => {
  return history.goForward();
};


const checkThis = (theThis) => {
  if (!theThis || !theThis.props) {
    throw new Error('The this is not a component.');
  }
  if (!theThis.props.router) {
    throw new Error('please use withRouter.');
  }
};

const getThisBase = (theThis) => {
  if (theThis.props.location) {
    return theThis.props.location.pathname;
  } else if (typeof window !== 'undefined') {
    return window.location.pathname; // eslint-disable-line no-undef
  } else {
    throw new Error('can not find base path!');
  }
};

export const thisPush = (theThis, pathOrLoc) => {
  checkThis(theThis);
  const route = processPath(getThisBase(theThis), pathOrLoc);
  return theThis.props.router.push(route);
};

export const thisReplace = (theThis, pathOrLoc) => {
  checkThis(theThis);
  const route = processPath(getThisBase(theThis), pathOrLoc);
  return theThis.props.router.replace(route);
};

export const thisGo = (theThis, n) => {
  checkThis(theThis);
  return theThis.props.router.go(n);
};

export const thisGoBack = (theThis) => {
  checkThis(theThis);
  return theThis.props.router.goBack();
};

export const thisGoForward = (theThis) => {
  checkThis(theThis);
  return theThis.props.router.goForward();
};
