/* eslint-disable no-param-reassign */
/**
 * Created by yaohx_169 on 2017/6/8.
 */
import config, { cookie } from './config';
import { getCookie, setCookie, delCookie } from './helper';
import db from './db';

const getSubValue = (key, subKey) => {
  const value = getCookie(key);
  if (!value) {
    return null;
  }
  try {
    const json = JSON.parse(value);
    return json[subKey];
  } catch (e) {
    delCookie(key);
    throw new Error(`损坏的${key} cookie!`);
  }
};

const setSubValue = (key, subKey, value) => {
  const v = getCookie(key);
  if (v) {
    try {
      const json = JSON.parse(v);
      json[subKey] = value;
      setCookie(key, JSON.stringify(json));
    } catch (e) {
      delCookie(key);
      throw new Error(`损坏的${key} cookie!`);
    }
  } else {
    setCookie(key, JSON.stringify({
      [subKey]: value,
    }));
  }
};

const delSubValue = (key, subKey) => {
  const value = getCookie(key);
  if (value) {
    try {
      const json = JSON.parse(value);
      delete json[subKey];
      setCookie(key, JSON.stringify(json));
    } catch (e) {
      delCookie(key);
      throw new Error(`损坏的${key} cookie!`);
    }
  }
};

const getProductValue = (key) => {
  return getSubValue(key, config.productId);
};

const setProductValue = (key, value) => {
  setSubValue(key, config.productId, value);
};

const delProductValue = (key) => {
  delSubValue(key, config.productId);
};

const getUserValue = (key) => {
  const uid = getProductValue(cookie.userId);
  if (uid) {
    return getSubValue(key, `${config.productId}/${uid}`);
  } else {
    return null;
  }
};

const setUserValue = (key, value) => {
  const uid = getProductValue(cookie.userId);
  if (uid) {
    setSubValue(key, `${config.productId}/${uid}`, value);
  }
};

const delUserValue = (key) => {
  const uid = getProductValue(cookie.userId);
  if (uid) {
    delSubValue(key, `${config.productId}/${uid}`);
  }
};

export async function getToken() {
  return getProductValue(cookie.token);
}

export async function setToken(token) {
  setProductValue(cookie.token, token);
}

export async function delToken() {
  delProductValue(cookie.token);
}

export async function getUser() {
  const id = getProductValue(cookie.userId);
  return id ? {
    id,
    name: getProductValue(cookie.userName),
  } : null;
}

export async function setUser(id, name) {
  setProductValue(cookie.userId, id);
  setProductValue(cookie.userName, name);
}

export async function delUser() {
  delProductValue(cookie.userId);
  delProductValue(cookie.userName);
}

export async function getDomain() {
  const path = getUserValue(cookie.domainPath);
  return path ? {
    name: getUserValue(cookie.domainName),
    path,
  } : null;
}

export async function setDomain(name, path) {
  setUserValue(cookie.domainName, name);
  setUserValue(cookie.domainPath, path);
}

export async function delDomain() {
  delUserValue(cookie.domainName);
  delUserValue(cookie.domainPath);
}

export async function isAuthed() {
  return getToken().then(result => !!result);
}

export async function hasDomain() {
  return getDomain().then(result => !!result);
}

const normHistory = size => (history) => {
  if (!history) {
    history = {};
  }
  if (!history.size) {
    if (size) {
      history.size = size;
    } else {
      history.size = 10;
    }
  } else if (size && history.size !== size) {
    history.size = size;
  }
  if (history.size < 1) {
    history.size = 1;
  }
  if (!history.data) {
    history.data = [];
    history.start = 0;
    history.top = 0;
    history.empty = true;
  }
  return history;
};

const normData = key => (history) => {
  if (!key) {
    return history;
  }
  if (!history[key]) {
    history[key] = {};
  }
  if (!history[key].data) {
    history[key].data = [];
    history[key].start = 0;
    history[key].top = 0;
    history[key].empty = true;
  }
  return history;
};

const selectData = key => history => (key ? history[key] : history);

const next = (i, size) => {
  if (i < 0 || i >= size) {
    throw new Error(`out of range: ${i} in ${size}`);
  }
  if (i + 1 >= size) {
    return 0;
  } else {
    return i + 1;
  }
};

const prev = (i, size) => {
  if (i < 0 || i >= size) {
    throw new Error(`out of range: ${i} in ${size}`);
  }
  if (i - i < 0) {
    return size - 1;
  } else {
    return i - 1;
  }
};

export const histories = {
  async getLatest(name, uid) {
    return db([config.productId, 'history', name])(normHistory(), normData(uid), (history) => {
      const target = selectData(uid)(history);
      if (target.empty) {
        return null;
      }
      return target.data[prev(target.top, history.size)];
    });
  },
  async createHistory(name, uid, size) {
    return db([config.productId, 'history', name]).write(normHistory(size), normData(uid));
  },
  async destroyHistory(name, uid) {
    const path = [config.productId, 'history', name];
    if (uid) {
      path.push(uid);
    }
    return db(path).delete();
  },
  async getHistory(name, uid, size) {
    return db([config.productId, 'history', name])(normHistory(size), normData(uid), (h) => {
      const target = selectData(uid)(h);
      if (target.empty) {
        return [];
      } else if (target.top > target.start) {
        return target.data.slice(target.start, target.top);
      } else {
        return [...target.data.slice(target.start, h.size), ...target.data.slice(0, target.top)];
      }
    });
  },
  async pushHistory(name, uid, value, size) {
    return db([config.productId, 'history', name]).write(normHistory(size), normData(uid), (history) => {
      const target = selectData(uid)(history);
      target.data[target.top] = value;
      const nextPos = next(target.top, history.size);
      if (!target.empty && target.start === target.top) {
        target.top = target.start = nextPos;
      } else {
        target.top = nextPos;
      }
      if (target.empty) {
        target.empty = false;
      }
      return history;
    });
  },
  async popHistory(name, uid) {
    return db([config.productId, 'history', name]).write(normHistory(), normData(uid), (history) => {
      const target = selectData(uid)(history);
      if (target.empty) {
        return;
      }
      target.top = prev(target.top, history.size);
      if (target.top === target.start) {
        target.empty = true;
      }
      return history;
    });
  },
  async init() {
    return db.read();
  },
};
