/* eslint-disable no-param-reassign */
/** @module utils/db */
import set from 'lodash/set';
import unset from 'lodash/unset';
import flow from 'lodash/fp/flow';
import getOr from 'lodash/fp/getOr';
import LocalStorage from 'lowdb/adapters/LocalStorage';
import { isPromise } from './helper';

// eslint-disable-next-line no-underscore-dangle
const _adapter_ = new LocalStorage('db');

/**
 * @param {CreateDB} db
 * @param key
 * @param adapter
 * @return {*}
 */
const init = function init(db, key, adapter) {
  db.read = () => {
    const r = adapter.read();

    return isPromise(r) ? r.then(db.plant) : db.plant(r);
  };

  db.write = (...args) => {
    const value = args.length > 0 && args[0] !== undefined ? args[0] : db.getState();

    const w = adapter.write(db.getState());

    return isPromise(w) ? w.then(() => {
      return value;
    }) : value;
  };

  db.plant = (state) => {
    db[key] = state;
    return db;
  };

  db.getState = () => {
    return db[key];
  };

  db.setState = (state) => {
    db.plant(state);
    return db;
  };

  return db.read();
};

/**
 * @typedef {Function} CreateDB
 * @param {string} path
 * @param {*} [defaultValue]
 * @return {DB}
 */

/**
 * @member {Function} CreateDB~read
 * @return {*|Promise.<*>}
 */

/**
 * @member {Function} CreateDB~write
 * @param {*} [state]
 * @return {*|Promise.<*>}
 */

/**
 * @member {Function} CreateDB~plant
 * @param {*} state
 * @return {CreateDB}
 */

/**
 * @member {Function} CreateDB~getState
 * @return {*}
 */

/**
 * @member {Function} CreateDB~setState
 * @param {*} state
 * @return {CreateDB}
 */

/**
 * @typedef {Function} DB
 * @param {...Function} functions
 * @return {*}
 */

/**
 * @member {Function} DB~write
 * @param {...Function} functions
 * @return {*|Promise.<*>}
 */

/**
 * @member {Function} DB~delete
 * @return {*|Promise.<*>}
 */

/**
 * @return {CreateDB}
 */
const CreateDB = () => {
  function db(path, defaultValue) {
    function getValue(...functions) {
      const result = getOr(defaultValue, path, db.getState());
      return flow(...functions)(result);
    }

    getValue.write = (...args) => {
      const result = getValue(...args);
      set(db.getState(), path, result);
      return db.write();
    };
    getValue.delete = (flush = true) => {
      unset(db.getState(), path);
      return flush ? db.write() : db.getState();
    };

    return getValue;
  }

  return init(db, '__state__', _adapter_);
};
export default CreateDB();

