import React from 'react';
import { message, Modal } from 'antd';
import debounce from 'lodash/debounce';
import memoize from 'lodash/memoize';
import { push } from '../services/route';
import { errors as errorCodes } from './config';

const errStyle = {
  overflow: 'auto',
  wordWrap: 'break-word',
  width: '360px',
  height: '360px',
  marginLeft: '-24px',
  marginTop: '24px',
};

function memDebounce(func, wait = 0, options = {}) {
  const mem = memoize(() => {
    const memFunc = (...args) => {
      const ret = func(...args);
      const { cache } = mem;
      cache.delete(args[0]);
      return ret;
    };
    // eslint-disable-next-line lodash-fp/no-extraneous-args
    return debounce(memFunc, wait, options);
  });
  return (...args) => mem(...args)(...args);
}

const msgError = memDebounce((msg) => {
  message.error(msg);
}, 300);

export function processError(err) {
  if (err) {
    if (err.data && err.data.errorCode) {
      const data = err.data;
      switch (data.errorCode) {
        case errorCodes.no_such_user:
          msgError('用户不存在！');
          break;
        case errorCodes.invalid_token:
        case errorCodes.token_missing:
          push('/login');
          break;
        case errorCodes.no_domain_right:
          msgError('没有此作用域权限。');
          push('/domain');
          break;
        default:
          showError(err);
      }
    } else if (err instanceof Error) {
      showError(err);
    }
  }
}

function showError(err) {
  if (err) {
    if (err.data && err.data.errorCode) {
      let msg;
      if (err.data) {
        const data = err.data;
        msg = data ? data.message : err.message;
      } else {
        msg = err.message;
      }
      if (msg && msg.length < 256) {
        msgError(msg);
      } else {
        Modal.error({
          title: '服务器内部错误',
          content: <div style={errStyle}>{msg}</div>,
          width: 460,
        });
      }
    } else if (err instanceof Error) {
      Modal.error({
        title: '错误！',
        content: (
          <div style={errStyle}>
            <p>{err.message ? err.message : ''}</p>
            <p>{err.stack ? err.stack : ''}</p>
          </div>
        ),
        width: 460,
      });
    }
  }
}

export function createError({ code, msg }) {
  const error = new Error(msg);
  error.data = {
    errorCode: code,
    message: msg,
  };
  return error;
}

export const errors = {
  tokenMissing: () => createError({
    code: errorCodes.token_missing,
    msg: '需要登录！',
  }),
  wrongPassword: () => createError({
    code: errorCodes.wrong_password,
    msg: '密码错误！',
  }),
  authFailed: msg => createError({
    code: errorCodes.auth_failed,
    msg: `登录验证失败！${msg || ''}`,
  }),
  unsupportedAuthType: (...types) => createError({
    code: errorCodes.unsupported_auth_type,
    msg: `不支持的客户端验证方式：${types.join('、')}.`,
  }),
  generalError: msg => createError({
    code: errorCodes.general_error,
    msg: `${msg || '未知错误。'}`,
  }),
};
