import post from '../utils/post';
import doDelete from '../utils/delete';
import config from '../utils/config';
import request from '../utils/request';
import { getToken } from '../utils/auth';
import { encrypt } from '../utils/helper';
import { errors } from '../utils/error';

export class Operations {
  constructor() {
    this.operations = [];
  }
  addCmd = (cmd, ...args) => {
    this.operations.push({ cmd, args });
    return this;
  };
  setName = (name) => {
    return this.addCmd('setName', name);
  };
  setRight = (right) => {
    return this.addCmd('setRight', right);
  };
  addTags = (...tags) => {
    return this.addCmd('addTags', ...tags);
  };
  removeTags = (...tags) => {
    return this.addCmd('removeTags', ...tags);
  };
  clearTags = () => {
    return this.addCmd('clearTags');
  };
  setTags = (...tags) => {
    return this.addCmd('setTags', ...tags);
  };
  addOwners = (...owners) => {
    return this.addCmd('addOwners', ...owners);
  };
  removeOwners = (...owners) => {
    return this.addCmd('removeOwners', ...owners);
  };
  clearOwners = () => {
    return this.addCmd('clearOwners');
  };
  setOwners = (...owners) => {
    return this.addCmd('setOwners', ...owners);
  };
  addOwnerGroups = (...groups) => {
    return this.addCmd('addOwnerGroups', ...groups);
  };
  removeOwnerGroups = (...groups) => {
    return this.addCmd('removeOwnerGroups', ...groups);
  };
  clearOwnerGroups = () => {
    return this.addCmd('clearOwnerGroups');
  };
  setOwnerGroups = (...groups) => {
    return this.addCmd('setOwnerGroups', ...groups);
  };
  use = (usage) => {
    return this.addCmd('use', usage);
  };
  unuse = (usage) => {
    return this.addCmd('unuse', usage);
  };
}

export class Query {
  constructor() {
    this.pageSize = -1;
  }
  setAfter = (after) => {
    this.after = after;
  };
  /**
   * create_time: 创建时间,
   * media_type: 媒体类型，比如application/pdf,
   * size: 文件大小,
   * name: 文件名称,
   * modify_time: 上一次修改时间,
   * uri: 资源定位符
   */
  setOrder = () => {
    const argn = arguments.length;
    if (argn > 0) {
      this.orders = [];
      let lastOrder;
      for (let i = 0; i < argn; ++i) {
        if (lastOrder !== undefined && (typeof arguments[i] === 'boolean')) {
          this.orders.push({
            target: lastOrder,
            asc: arguments[i],
          });
          lastOrder = undefined;
        } else if (lastOrder !== undefined) {
          this.orders.push({
            target: lastOrder,
            asc: true,
          });
          lastOrder = arguments[i];
        } else {
          lastOrder = arguments[i];
        }
      }
      if (lastOrder !== undefined) {
        this.orders.push({
          target: lastOrder,
          asc: true,
        });
      }
    }
  };
  setPageSize = (size) => {
    this.pageSize = size >= 0 ? size : -1;
  };
  setCondition = (condition) => {
    this.condition = condition;
  };
  setStores = (...stores) => {
    this.stores = stores;
  };
  static tag = (...tags) => {
    if (tags.length === 0) {
      throw new Error('At least one tag!');
    }
    if (tags.length === 1) {
      return {
        type: 'tag',
        value: tags[0],
      };
    } else {
      return Query.and(...tags.map(tag => Query.tag(tag)));
    }
  };
  static usage = (...usage) => {
    if (usage.length === 0) {
      throw new Error('At least one usage!');
    }
    if (usage.length === 1) {
      return {
        type: 'usage',
        value: usage[0],
      };
    } else {
      return Query.and(...usage.map(tag => Query.usage(tag)));
    }
  };
  static and = (...conditions) => {
    return {
      type: 'and',
      conditions,
    };
  };
  static or = (...conditions) => {
    return {
      type: 'or',
      conditions,
    };
  };
}

export const createOperations = () => {
  return new Operations();
};

export async function editResource(uri, operations) {
  return post(`${config.apiContextPath}/api/resource/user/${encodeURIComponent(uri)}/meta`, { operations: operations.operations });
}

export async function deleteResource(uri) {
  return doDelete(await rsLink(uri));
}

export async function countResource(query) {
  return post(`${config.apiContextPath}/api/resource/user/count`, query);
}

export async function queryResource(query) {
  return post(`${config.apiContextPath}/api/resource/user/query`, query);
}

export async function getResourceInfo(uri) {
  return request(`${config.apiContextPath}/api/resource/user/${encodeURIComponent(uri)}/meta`);
}

export async function rsLink(uri, download = false) {
  const token = await getToken();
  if (!token) {
    throw errors.tokenMissing();
  }
  return `${config.apiContextPath}/resource/${encrypt(token)}/${encodeURIComponent(uri)}${download ? '?dl=true' : ''}`;
}
